% getEventFraction.mn
%
% This script takes in two data sets generated by the Behavior Analyzer,
% which is a list of times when an event occurs.  By giving this script two
% lists of events, it will return the fraction of events of the first type
% of the total number of events indicated by the second type over a
% specified time interval.  It will return a vector if a vector of lists is
% provided (which is how it's supposed to be used, so these data can be
% dropped into Prism to produce bar graphs with standard error bars).
%
% To be the most fair, we ignore both numerator and denominator events in
% intervals where either has missing data.
%
% Because some worms were only observed for a brief period of time during
% light exposure (e.g. they ran away), we also provide a final pooled
% statistic which calculates the fraction by pooling all the data together and then calulating the fraction.
% For example, one I1- worm was observed for 1 sec or so before it
% disappeared, and didn't spit at all during these pumps.  This one worm
% would weight the average down because its fraction will be 0, but this
% doesn't seem right.  So I will pool the data as well.

function [fractions pooledFraction numPooled denPooled numPerWorm weightedNumRate weights] = ...
    getEventFraction(numeratorEvents, numeratorMisses, ...
    denominatorEvents, denominatorMisses, startInSec, stopInSec)

fractions = zeros(length(numeratorEvents), 1);
startInMs = startInSec * 1000;
stopInMs = stopInSec * 1000;

numPooled = 0;
denPooled = 0;

numPerWorm = zeros(length(numeratorEvents), 1);
weights = zeros(length(numeratorEvents), 1);

for i=1:length(fractions)
    numEvents = numeratorEvents{i}(numeratorEvents{i} >= startInMs & numeratorEvents{i} <= stopInMs);
    numMissing = intersect(numEvents, denominatorMisses{i});
    num = length(numEvents) - length(numMissing);

    denEvents = denominatorEvents{i}(denominatorEvents{i} >= startInMs & denominatorEvents{i} <= stopInMs);
    denMissing = intersect(denEvents, numeratorMisses{i});
    den = length(denEvents) - length(denMissing);

    fractions(i) = num / den;
    numPerWorm(i) = num;
    
    numPooled = numPooled + num;
    denPooled = denPooled + den;

    numMiss = length(find(numeratorMisses{i} >= startInMs & numeratorMisses{i} <= stopInMs));
    denMiss = length(find(denominatorMisses{i} >= startInMs & denominatorMisses{i} <= stopInMs));

    weights(i) = 1 - max(numMiss, denMiss) / (stopInMs - startInMs);
end

pooledFraction = numPooled / denPooled;
weightedNumRate = sum(numPerWorm .* weights) / sum(weights) / (stopInSec - startInSec);

end